const core = require('@sentry/core');
const node = require('@sentry/node');
const electron = require('electron');
const envelope = require('../common/envelope.js');
const ipc = require('../common/ipc.js');
const merge = require('./merge.js');

function _interopNamespaceDefault(e) {
    const n = Object.create(null);
    if (e) {
        for (const k in e) {
            n[k] = e[k];
        }
    }
    n.default = e;
    return n;
}

const electron__namespace = /*#__PURE__*/_interopNamespaceDefault(electron);

function log(message) {
    core.debug.log(`[Utility Process] ${message}`);
}
/**
 * We wrap `electron.utilityProcess.fork` so we can pass a messageport to any SDK running in the utility process
 */
function configureUtilityProcessIPC() {
    if (!electron__namespace.utilityProcess?.fork) {
        return;
    }
    // eslint-disable-next-line @typescript-eslint/unbound-method
    electron__namespace.utilityProcess.fork = new Proxy(electron__namespace.utilityProcess.fork, {
        apply: (target, thisArg, args) => {
            // Call the underlying function to get the child process
            const child = target.apply(thisArg, args);
            function getProcessName() {
                const [, , options] = args;
                return options?.serviceName || `pid:${child.pid}`;
            }
            // We don't send any messages unless we've heard from the child SDK. At that point we know it's ready to receive
            // and will also filter out any messages we send so users don't see them
            child.on('message', (msg) => {
                if (ipc.isMagicMessage(msg)) {
                    log(`SDK started in utility process '${getProcessName()}'`);
                    const { port1, port2 } = new electron__namespace.MessageChannelMain();
                    port2.on('message', (msg) => {
                        if (msg.data instanceof Uint8Array || typeof msg.data === 'string') {
                            handleEnvelopeFromUtility(msg.data);
                        }
                    });
                    port2.start();
                    // Send one side of the message port to the child SDK
                    child.postMessage(ipc.getMagicMessage(), [port1]);
                }
            });
            // We proxy child.on so we can filter messages from the child SDK and ensure that users do not see them
            // eslint-disable-next-line @typescript-eslint/unbound-method
            child.on = new Proxy(child.on, {
                apply: (target, thisArg, [event, listener]) => {
                    if (event === 'message') {
                        return target.apply(thisArg, [
                            'message',
                            (msg) => {
                                if (ipc.isMagicMessage(msg)) {
                                    return;
                                }
                                return listener(msg);
                            },
                        ]);
                    }
                    return target.apply(thisArg, [event, listener]);
                },
            });
            return child;
        },
    });
}
function handleEnvelopeFromUtility(env) {
    const envelope$1 = core.parseEnvelope(env);
    const eventAndAttachments = envelope.eventFromEnvelope(envelope$1);
    if (eventAndAttachments) {
        const [event, attachments] = eventAndAttachments;
        captureEventFromUtility(event, attachments);
    }
    else {
        // Pass other types of envelope straight to the transport
        void node.getClient()?.getTransport()?.send(envelope$1);
    }
}
function captureEventFromUtility(event, attachments) {
    // Remove the environment as it defaults to 'production' and overwrites the main process environment
    delete event.environment;
    delete event.release;
    // Remove the SDK info as we want the Electron SDK to be the one reporting the event
    delete event.sdk?.name;
    delete event.sdk?.version;
    delete event.sdk?.packages;
    node.captureEvent(merge.mergeEvents(event, { tags: { 'event.process': 'utility' } }), { attachments });
}

exports.configureUtilityProcessIPC = configureUtilityProcessIPC;//# sourceMappingURL=http://go/sourcemap/sourcemaps/2ca326e0d1ce10956aea33d54c0e2d8c13c58a30/node_modules/@sentry/electron/main/utility-processes.js.map
